#!/usr/bin/python -tt

#    Two
#    Copyright (C) 2011 Segfault Garden
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.


import gtk, cairo, math, time, gobject, gst ,os, sys
from math import cos, sin, atan

class Point(object):
    __slots__ = ['x', 'y']
    
    def __init__(self, x, y):
        self.x = x
        self.y = y
        
    def __str__(self):
        return '(%f, %f)' % (self.x, self.y)
    
    def dot_product(self, other):
        return self.x*other.x + self.y*other.y
    
    def minus(self, other):
        return Point(self.x - other.x, self.y - other.y)
    
    def plus(self, other):
        return Point(self.x + other.x, self.y + other.y)
        
    def length(self):
        return math.sqrt(self.dot_product(self))
    
    def divide(self, value):
        assert(value != 0)
        return Point(self.x/value, self.y/value)
    
    def times(self, value):
        return Point(self.x*value, self.y*value)
    
    def normalized(self):
        l = self.length()
        if l == 0:
            return self
        return self.divide(l)

def determinant(row1_column1, row1_column2, row2_column1, row2_column2):
    return row1_column1*row2_column2 - row1_column2*row2_column1

def line_intersection(p1, p2, p3, p4):
    (x, y) = line_intersection_unpacked(p1.x, p1.y,
                                        p2.x, p2.y,
                                        p3.x, p3.y,
                                        p4.x, p4.y)
    return Point(x, y)

def line_intersection_unpacked(x1, y1, x2, y2, x3, y3, x4, y4):
    r1c1 = determinant(x1, y1, x2, y2)
    r1c2 = x1-x2
    r2c1 = determinant(x3, y3, x4, y4)
    r2c2 = x3 - x4
    
    numerator = determinant(r1c1, r1c2, r2c1, r2c2)
    denominator = determinant(x1 - x2, y1 - y2, x3 - x4, y3 - y4)
    x = numerator / denominator
    
    r1c2 = y1-y2
    r2c2 = y3 - y4
    
    numerator = determinant(r1c1, r1c2, r2c1, r2c2)
    y = numerator / denominator
    
    return (x, y)

def subdivide_line(p1, p2, horizont, subdivisions):
        station = Point(90, horizont)

        if p2.y > p1.y:
            (p1, p2) = (p2, p1)
        
        if p2.x > p1.x:
            side_off = 10
        else:
            side_off = -10
        sideways = Point(p1.x+side_off, p1.y)
        extended = line_intersection(station, p2, p1, sideways)
        delta_x = extended.x - p1.x
        result = []
        for i in subdivisions:
            cur_x = p1.x + i*delta_x
            cur_y = p1.y
            cur = Point(cur_x, cur_y)
            intersection = line_intersection(station, cur, p1, p2)
            result.append(intersection)
        return result

def extend_line(p1, p2):
    """Returns point p3 that projected distance from p1 to p3 is
    double that of distance p1 to p2."""
    if abs(p1.y - p2.y) < 0.01:
        return Point(p1.x + 2*(p2.x-pi.x))
    up1 = Point(p1.x, 0)
    up2 = Point(p2.x, p2.y/2)
    p3 = line_intersection(p1, p2, up1, up2)
    return p3

def project_cube(p0, p1, p2, cube_left, cube_right, height, horizont, lvp, rvp):
    assert(p1.x > p2.x)
    assert(p1.y <= p0.y)
    assert(p2.y <= p0.y)
    assert(lvp.y == horizont)
    assert(rvp.y == horizont)
    leftdivs = subdivide_line(p0, p2, horizont, cube_left)
    rightdivs = subdivide_line(p0, p1, horizont, cube_right)
    assert(leftdivs[0].x >= leftdivs[1].x)
    assert(rightdivs[0].x <= rightdivs[1].x)
    c0 = line_intersection(leftdivs[0], rvp, rightdivs[0], lvp)
    c1 = line_intersection(leftdivs[0], rvp, rightdivs[1], lvp)
    c2 = line_intersection(leftdivs[1], rvp, rightdivs[0], lvp)
    c3 = Point(c0.x, c0.y - height)
    c1_up = Point(c1.x, c1.y-10)
    c2_up = Point(c2.x, c2.y-10)
    c4 = line_intersection(c1, c1_up, c3, rvp)
    c5 = line_intersection(c2, c2_up, c3, lvp)
    assert(c4.y < c3.y)
    assert(c5.y < c3.y)
    return [c0, c1, c2, c3, c4, c5]

def corner_cube_to_full_cube(projected_cube, lvp, rvp):
    down_back = line_intersection(projected_cube[1], lvp, projected_cube[2], rvp)
    up_back = line_intersection(projected_cube[4], lvp, projected_cube[5], rvp)
    bottom = (projected_cube[0], projected_cube[1], down_back, projected_cube[2])
    top = (projected_cube[3], projected_cube[4], up_back, projected_cube[5])
    return (bottom, top)
    

def get_perp_point(vp1, vp2, vp3):
    dx = vp2.x - vp3.x
    dy = vp2.y - vp3.y
    direction_point = Point(vp1.x - dy, vp1.y + dx)
    perp_point = line_intersection(vp3, vp2, vp1, direction_point)
    return perp_point

def evaluate_magic_spot(l, r, b, p):
    # Son, I am disappoint.
    min_dp = 1e200
    min_t = 0
    for i in range(1000):
        frac = i/1000.0
        testpoint = Point(frac*b.x + (1-frac)*p.x, frac*b.y + (1-frac)*p.y)
        d1 = l.minus(testpoint)
        d2 = r.minus(testpoint)
        dp = d1.dot_product(d2)
        if abs(dp) < abs(min_dp):
            min_dp = dp
            min_t = frac
    t = min_t
    
    res = Point(t*b.x + (1-t)*p.x, t*b.y + (1-t)*p.y)
    return res

def solve_dvp(lvp, rvp, magic):
    d1 = lvp.minus(magic).normalized()
    d2 = rvp.minus(magic).normalized()
    off = magic.plus(d1).plus(d2)
    dvp = line_intersection(lvp, rvp, magic, off)
    return dvp

def build_threecube(points, vps, dvps):
    p0 = line_intersection(points[0], vps[2], points[1], dvps[1])
    p1 = line_intersection(points[1], vps[2], p0, vps[1])
    p3 = line_intersection(points[-1], vps[2], p0, vps[0])
    p2 = line_intersection(p1, vps[0], p3, vps[1])
    return [p0, p1, p2, p3]

class SimplePerspective():
    
    def __init__(self):
        self.horizont = 300
        self.vp = 400
        self.left_dvp = 0
        self.right_dvp = 800

class PerspectiveBuilder():
    
    def __init__(self, magic, horizont):
        self.magic = magic
        self.horizont = horizont
        assert(magic.y > horizont)
        
    def build_1d(self):
        vp = Point(self.magic.x, self.horizont)
        (ldvp, vp, rdvp) = self.build_2d(math.pi/4)
        # The order is the same, but the names are different
        # to emphasize that the semantics change.
        return (ldvp, vp, rdvp)
        
    def build_2d(self, angle):
        assert(angle >= 0.0)
        assert(angle <= math.pi/2)
        
        h1 = Point(0, self.horizont)
        h2 = Point(10, self.horizont)
        
        d = 10.0
        
        ltmp = Point(self.magic.x - d*cos(angle), self.magic.y - d*sin(angle))
        lvp = line_intersection(self.magic, ltmp, h1, h2)
        
        rtmp = Point(self.magic.x + d*cos(math.pi/2-angle), self.magic.y - d*sin(math.pi/2-angle))
        rvp = line_intersection(self.magic, rtmp, h1, h2)
        
        dtmp = Point(0.5*(rtmp.x +ltmp.x), 0.5*(rtmp.y + ltmp.y))
        dvp = line_intersection(self.magic, dtmp, h1, h2)
        
        return (lvp, dvp, rvp)

class PlanProjector():
    
    def __init__(self):
        self.picture_plane = 300
        self.horizont = 400
        self.ground = 500
        self.square_size = 100
        self.station = Point(400, 0) # Dummy, but x coordinate is correct
        self.set_theta(math.pi/4)
        
    def get_points(self):
        theta = self.theta
        gamma = math.pi/2 - theta
        p0 = Point(self.station.x, self.picture_plane)
        p1 = Point(p0.x + self.square_size*cos(gamma), p0.y - self.square_size*sin(gamma))
        p2 = Point(p0.x - self.square_size*cos(theta), p0.y - self.square_size*sin(theta))
        return [p0, p1, p2]
    
    def set_theta(self, theta):
        if theta < 0.01:
            theta = 0.01
        if theta > math.pi/2-0.01:
            theta = math.pi/2-0.01
        self.theta = theta
        self.locate_station()
        self.project_edges()
        
    def locate_station(self):
        width = self.square_size*(cos(self.theta) + cos(math.pi/2.0 - self.theta))
        self.station = Point(self.station.x, self.picture_plane + 2*width)
        
    def project_edges(self):
        theta = self.theta
        gamma = math.pi/2 - theta
        pp1 = Point(0, self.picture_plane)
        pp2 = Point(1000, self.picture_plane)
        hp1 = Point(0, self.horizont)
        hp2 = Point(1000, self.horizont)
        
        d1 = Point(self.station.x - self.square_size*cos(theta), self.station.y - self.square_size*sin(theta))
        d2 = Point(self.station.x + self.square_size*cos(gamma), self.station.y - self.square_size*sin(gamma))
        
        a = line_intersection(pp1, pp2, self.station, d1)
        b = line_intersection(pp1, pp2, self.station, d2)
        self.lvp = Point(a.x, self.horizont)
        self.rvp = Point(b.x, self.horizont)
        self.magic_spot = Point(self.station.x, self.station.y + (self.horizont - self.picture_plane))
        
        #diag_point = Point(self.lvp.x + self.rvp.x - self.magic_spot.x,
        #                   self.lvp.y + self.rvp.y - self.magic_spot.y)
        lstep_x = self.lvp.x - self.magic_spot.x
        lstep_y = self.lvp.y - self.magic_spot.y
        angle = atan(abs(lstep_y)/abs(lstep_x))

        d = 10
        ltmp = Point(self.magic_spot.x - d*cos(angle), self.magic_spot.y - d*sin(angle))
        rtmp = Point(self.magic_spot.x + d*cos(math.pi/2-angle), self.magic_spot.y - d*sin(math.pi/2-angle))
        dtmp = Point(0.5*(rtmp.x +ltmp.x), 0.5*(rtmp.y + ltmp.y))
        self.dvp = line_intersection(self.magic_spot, dtmp, hp1, hp2)

    def get_vps(self):
        return (self.lvp, self.dvp, self.rvp)

class TwoApp():
    
    def __init__(self, fullscreen=False):
        self.width = 800
        self.height = 600
        self.fullscreen = fullscreen
        self.build_gui()
        self.persp = SimplePerspective()
        magic_spot = Point(self.width/2, 500)
        self.pbuild = PerspectiveBuilder(magic_spot, self.persp.horizont)
        self.player = gst.element_factory_make('playbin', ' player')
        self.player.set_property('uri', 'file://' + os.getcwd() + '/SG-Two.ogg')
        self.player.set_state(gst.STATE_PLAYING)
        self.zero_time = time.time()
        
    def build_gui(self):
        self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
        self.window.set_title('Segfault Garden: Two')
        self.window.set_resizable(False)
        if self.fullscreen:
            self.window.fullscreen()
        self.window.connect('destroy', gtk.main_quit)
        self.window.connect('key-press-event', self.key_pressed)
        self.canvas = gtk.DrawingArea()
        self.canvas.connect('expose-event', self.expose_canvas)
        self.canvas.set_size_request(self.width, self.height)
        self.canvas.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse("white"))
        self.window.add(self.canvas)
        self.window.show_all()
        gobject.timeout_add(20, self.delay_callback)
        
    def delay_callback(self):
        self.canvas.queue_draw()
        return True
        
    def expose_canvas(self, widget, data):
        #cur_time = time.time() - self.zero_time
        #self.seg_time = cur_time
        #self.fill_test(widget, data)
        #self.shadow_test(widget, data)
        #self.intersection_test(widget, data)
        #self.text_test(widget, data)
        #self.three_point_test(widget, data)
        #self.corner_test(widget, data)
        #self.dot_test(widget, data)
        #self.line_test(widget, data)
        self.main_loop(widget, data)
        
    def main_loop(self, widget, data):
        cur_time = time.time() - self.zero_time
        engtext = ['An engine', 'without:']
        if cur_time < 10:
            self.seg_time = cur_time
            self.title(widget, data)
        elif cur_time < 15:
            self.seg_time = cur_time - 10
            self.intertitle(widget, data, engtext, ['Hardware', 'acceleration'])
        elif cur_time < 25:
            self.seg_time = cur_time - 20
            self.dot_test(widget, data)
        elif cur_time < 30:
            self.seg_time = cur_time - 25
            self.intertitle(widget, data, engtext, ['3D projection math'])
        elif cur_time < 40:
            self.seg_time = cur_time - 30
            self.line_test(widget, data)
        elif cur_time < 45:
            self.seg_time = cur_time - 40
            self.intertitle(widget, data, engtext, ['Backface culling'])
        elif cur_time < 55:
            self.seg_time = cur_time - 45
            self.corner_test(widget, data)
        elif cur_time < 60:
            self.seg_time = cur_time - 55
            self.intertitle(widget, data, engtext, ['Matrices'])
        elif cur_time < 70:
            self.seg_time = cur_time - 60
            self.fill_test(widget, data)
        elif cur_time < 75:
            self.seg_time = cur_time - 70
            self.intertitle(widget, data, engtext, ['Z buffering', 'Z sorting'])
        elif cur_time < 85:
            self.seg_time = cur_time - 75
            self.intersection_test(widget, data)
        elif cur_time < 90:
            self.seg_time = cur_time - 85
            self.intertitle(widget, data, engtext, ['Precalc'])
        elif cur_time < 100:
            self.seg_time = cur_time - 90
            self.three_point_test(widget, data)
        elif cur_time < 105:
            self.seg_time = cur_time - 100
            self.intertitle(widget, data, ['And even', 'without:'], ['Polygon intersections'])
        elif cur_time < 115:
            self.seg_time = cur_time - 105
            self.shadow_test(widget, data)
        elif cur_time < 122:
            self.seg_time = cur_time - 115
            self.end_title(widget, data)
        else:
            gtk.main_quit()
        
    def draw_array(self, cr, arr, fill=False):
        p = arr[0]
        cr.move_to(p.x, p.y)
        for p in arr[1:]:
            cr.line_to(p.x, p.y)
        cr.close_path()
        if fill:
            cr.fill()
        else:
            cr.stroke()
            
    def three_point_test(self, widget, data):
        now = time.time()
        duration = 500
        fraction = (int(100*now) % duration)/float(duration)
        theta = math.pi/4 + math.pi/12.0*sin(math.pi*2.0*fraction)
        pp = PlanProjector()
        #pp.square_size = 40
        #pp.horizont = 200
        pp.horizont = 100 #+ 30*math.sin(8*math.pi*fraction)
        pp.set_theta(theta)
        points = pp.get_points()
        (lvp, dummy, rvp) = pp.get_vps()
        projpoints = self.project_plan(points, pp, lvp, rvp)
        unit_square = self.build_square(projpoints, lvp, rvp)
#        unit_square = self.double_square(unit_square, lvp, rvp)

        cr = self.canvas.window.cairo_create()
        horizont = pp.horizont
        bottom_vp = Point((lvp.x + rvp.x)/2 + 100*sin(3*math.pi*fraction), self.height + 750 + 300*cos(5*math.pi*fraction))
        
        perp_point1 = get_perp_point(lvp, rvp, bottom_vp)
        perp_point2 = get_perp_point(rvp, bottom_vp, lvp)
        perp_point3 = get_perp_point(bottom_vp, lvp, rvp)

        self.set_color(cr, 0.7)
        magic1 = evaluate_magic_spot(lvp, rvp, bottom_vp, perp_point3)
        magic2 = evaluate_magic_spot(rvp, bottom_vp, lvp, perp_point1)
        magic3 = evaluate_magic_spot(bottom_vp, lvp, rvp, perp_point2)

        dvp1 = solve_dvp(lvp, rvp, magic1)
        dvp2 = solve_dvp(rvp, bottom_vp, magic2)
        dvp3 = solve_dvp(bottom_vp, lvp, magic3)

        vps = [lvp, rvp, bottom_vp]
        dvps = [dvp1, dvp2, dvp3]
        top = build_threecube(projpoints, vps, dvps)
        arr1 = [unit_square[0], unit_square[1], unit_square[3], top[0], top[1], top[3]]
        
        cube2_bottom = subdivide_line(projpoints[0], projpoints[1], horizont, [1.3, 2.3])
        newp = line_intersection(cube2_bottom[0], lvp, projpoints[-1], rvp)
        back = line_intersection(newp, rvp, cube2_bottom[1], lvp)
        cube2_bottom.append(back)
        cube2_bottom.append(newp)
        cube2_top = build_threecube(cube2_bottom, vps, dvps)
        arr2 = [cube2_bottom[0], cube2_bottom[1], cube2_bottom[3], cube2_top[0], cube2_top[1], cube2_top[3]]
        
        cube3_bottom = subdivide_line(projpoints[0], projpoints[-1], horizont, [1.3, 2.3])
        newp = line_intersection(cube3_bottom[0], rvp, projpoints[1], lvp)
        back = line_intersection(newp, lvp, cube3_bottom[1], rvp)
        cube3_bottom = [cube3_bottom[0], newp, back, cube3_bottom[1]]
        cube3_top = build_threecube(cube3_bottom, vps, dvps)
        arr3 = [cube3_bottom[0], cube3_bottom[1], cube3_bottom[3], cube3_top[0], cube3_top[1], cube3_top[3]]

        self.draw_corner_cube(cr, arr3, lvp, rvp, horizont, 0.2*(math.pi-theta)/math.pi*2, 0.8*theta/math.pi*2, 0.5)
        self.draw_corner_cube(cr, arr2, lvp, rvp, horizont, 0.2*(math.pi-theta)/math.pi*2, 0.8*theta/math.pi*2, 0.5)
        self.draw_corner_cube(cr, arr1, lvp, rvp, horizont, 0.2*(math.pi-theta)/math.pi*2, 0.8*theta/math.pi*2, 0.5)
        
    def title(self, widget, data):
        if self.seg_time < 0.5:
            return
        elif self.seg_time < 5:
            self.draw_interbox(['Segfault Garden'], 60, 200, ['Two'], 100, 400)
        elif self.seg_time < 5.5:
            return
        else:
            self.draw_interbox(['The time has come', 'for a totally new kind of', '3D engine!'], 60, 200, [], 0, 0)
            
    def end_title(self, widget, data):
        if self.seg_time < 0.1:
            return
        self.draw_interbox(['Code:', 'Satoris'], 60, 200, ['Music:', 'Jicamu'], 60, 400)

    def intertitle(self, widget, data, text1, text2):
        if self.seg_time > 0.1:
            self.draw_interbox(text1, 60, 150, text2, 75, 400)
            
        
    def text_test(self, widget, data):
        text1_size = 60
        text2_size = 75
        text1_start_y = 150
        text1_size = 60
        text1 = ['This is', 'number one']
        text2_start_y = 400
        text2 = ['NOT', 'USED']
        text2_size = 100
        self.draw_interbox(text1, text1_size, text1_start_y, text2, text2_size, text2_start_y)

    def draw_interbox(self, text1, text1_size, text1_start_y,
                      text2, text2_size, text2_start_y):
        cr = self.canvas.window.cairo_create()
        cr.select_font_face('Ubuntu')
        cr.set_font_size(text1_size)
        cr.set_source_rgb(0, 0, 0)
        self.draw_centered_text(cr, text1, text1_start_y)
        cr.set_font_size(text2_size)
        self.draw_centered_text(cr, text2, text2_start_y)

    def draw_centered_text(self, cr, text, text_start_y):
        (ascent, descent, height, max_x_advance, max_y_advance) = cr.font_extents()
        
        i = 0
        for t in text:
            (x_bearing, y_bearing, width, height, x_advance, y_advance) = cr.text_extents(t)
            cr.move_to((self.width - width)/2, text_start_y + i*height*1.3)
            i += 1
            cr.show_text(t)

    def intersection_test(self, widget, data):
        now = time.time()
        duration = 500
        fraction = (int(100*now) % duration)/float(duration)
        theta = math.pi*0.25*(sin(math.pi*2.0*fraction) + 1)
        s_leftcolor = [x/255.0 for x in [26, 56, 241]]
        s_rightcolor = [x/255.0 for x in [106, 127, 253]]
        s_topcolor = [x/255.0 for x in [175, 186, 255]]
        m_leftcolor = [x/255.0 for x in [21, 239, 5]]
        m_rightcolor = [x/255.0 for x in [115, 248, 105]]
        m_topcolor = [x/255.0 for x in [199, 251, 195]]
        cr = self.canvas.window.cairo_create()
        cr.set_operator(cairo.OPERATOR_SOURCE)
        cr.set_source_rgb(0, 0, 0)
        pp = PlanProjector()
        pp.square_size = 550
        pp.horizont = -100
        pp.set_theta(theta)
        points = pp.get_points()
        (lvp, dvp, rvp) = pp.get_vps()
        
        projpoints = self.project_plan(points, pp, lvp, rvp)
        unit_square = self.build_square(projpoints, lvp, rvp)

        # Stationary
        s_boxsize = 0.4
        s_leftboxstart = 0.3
        s_rightboxstart = 0.2
        s_boxheight = 180
        s_leftboxend = s_leftboxstart + s_boxsize
        s_rightboxend = s_rightboxstart + s_boxsize
        s_projected_cube = project_cube(unit_square[0], unit_square[1], unit_square[3],
                                       [s_leftboxstart, s_leftboxend], [s_rightboxstart, s_rightboxend],\
                                       s_boxheight, pp.horizont, lvp, rvp)

        # Moving
        m_boxwidth = 0.2
        m_boxdepth = 0.8
        m_leftboxstart = 0.4
        m_rightboxstart = 0.8*0.5*(sin(4*2*math.pi*fraction) + 0.8)
        m_boxheight = 60
        m_leftboxend = m_leftboxstart + m_boxwidth
        m_rightboxend = m_rightboxstart + m_boxdepth

        if m_rightboxend > s_rightboxend:
            projected_cube = project_cube(unit_square[0], unit_square[1], unit_square[3],
                                          [m_leftboxstart, m_leftboxend], [s_rightboxend, m_rightboxend],\
                                           m_boxheight, pp.horizont, lvp, rvp)
            (bottom, top) = corner_cube_to_full_cube(projected_cube, lvp, rvp)
            self.draw_filled_cube(cr, bottom, top, m_rightcolor, m_leftcolor, m_topcolor)
        
        (s_bottom, s_top) = corner_cube_to_full_cube(s_projected_cube, lvp, rvp)
        self.draw_filled_cube(cr, s_bottom, s_top, s_rightcolor, s_leftcolor, s_topcolor)
        
        if m_rightboxstart < s_rightboxstart:
            projected_cube = project_cube(unit_square[0], unit_square[1], unit_square[3],
                                            [m_leftboxstart, m_leftboxend], [m_rightboxstart, s_rightboxstart],\
                                             m_boxheight, pp.horizont, lvp, rvp)
            (bottom, top) = corner_cube_to_full_cube(projected_cube, lvp, rvp)
            self.draw_filled_cube(cr, bottom, top, m_rightcolor, m_leftcolor, m_topcolor)
            

    def shadow_test(self, widget, data):
        duration = 10.0
        fraction = self.seg_time/duration
        h1 = Point(0, self.persp.horizont)
        h2 = Point(1000, h1.y)
        sun = Point(100 + fraction*(self.width - 200), 0.4*(1-sin(math.pi*fraction)) *self.persp.horizont)
        sun_radius = 30
        sun_ground = Point(sun.x, self.persp.horizont)
        theta = math.pi/6
        leftcolor = [x/255.0 for x in [171, 57, 46]]
        rightcolor = [x/255.0 for x in [193, 97, 87]]
        topcolor = [x/255.0 for x in [251, 137, 125]]
        cr = self.canvas.window.cairo_create()
        cr.set_operator(cairo.OPERATOR_SOURCE)
        cr.set_source_rgb(0, 126/255.0, 246/255.0)
        cr.rectangle(0, 0, self.width, self.persp.horizont)
        cr.fill()
        cr.set_source_rgb(237/255.0, 224/255.0, 217/255.0)
        cr.rectangle(0, self.persp.horizont, self.width, self.height)
        cr.fill()
        cr.set_source_rgb(253/255.0, 253/255.0, 83/255.0)
        cr.arc(sun.x, sun.y, sun_radius, 0, 2*math.pi)
        cr.fill()
        
        lvp = Point(-200, self.persp.horizont)
        rvp = Point(self.width + 100, self.persp.horizont)
        p0 = Point(2.0*self.width/3, self.height - 20)
        p0_height = Point(p0.x, self.persp.horizont + 150)
        top = Point(p0.x, 100)
        cross1 = Point(self.width-100, p0.y)
        cross2 = Point(100, p0.y)
        p1 = line_intersection(top, cross1, p0, rvp)
        p2 = line_intersection(top, cross2, p0, lvp)
        cr.set_source_rgb(0, 0, 0)
        
        boxsize = 0.1
        cube_height = 20
        
        num_boxes = 7
        for i in range(num_boxes):
            cur_fraction = fraction + float(i)/num_boxes
            leftboxstart = (1-boxsize)*(cos(4*math.pi*cur_fraction) + 1)
            rightboxstart = (1-boxsize)*(sin(8*math.pi*cur_fraction) + 1)

            leftboxend = leftboxstart + boxsize
            rightboxend = rightboxstart + boxsize
            projected_cube = project_cube(p0, p1, p2, [leftboxstart, leftboxend], [rightboxstart, rightboxend],\
                                          cube_height, self.persp.horizont, lvp, rvp)
            cr.set_source_rgb(0, 0, 0)
            self.draw_corner_cube_shadow(cr, projected_cube, sun, sun_ground)
            self.draw_corner_cube(cr, projected_cube, lvp, rvp, self.persp.horizont, leftcolor, rightcolor, topcolor)

    def dot_test(self, widget, data):
        now = time.time()
        duration = 500
        fraction = (int(100*now) % duration)/float(duration)
        theta = math.pi/2.0*fraction
        cr = self.canvas.window.cairo_create()
        cr.set_operator(cairo.OPERATOR_SOURCE)
        cr.set_source_rgb(0, 0, 0)
        shift = Point(0, -100)
        pp = PlanProjector()
        pp.horizont = 400
        pp.set_theta(theta)
        points = pp.get_points()
        p = points[0]
        (lvp, dvp, rvp) = pp.get_vps()
        
        projpoints = self.project_plan(points, pp, lvp, rvp)
        unit_square = self.build_square(projpoints, lvp, rvp)
        big_square = self.double_square(unit_square, lvp, rvp)
        big_top = self.build_cube(big_square, 300, lvp, rvp)
        big_square = [x.plus(shift) for x in big_square]
        big_top = [x.plus(shift) for x in big_top]
        self.draw_dot_cube(cr, big_square, big_top)

    def line_test(self, widget, data):
        now = time.time()
        duration = 250
        fraction = (int(100*now) % duration)/float(duration)
        theta = math.pi/2.0*(1-fraction)
        cr = self.canvas.window.cairo_create()
        cr.set_operator(cairo.OPERATOR_SOURCE)
        cr.set_source_rgb(0, 0, 0)
        shift = Point(0, -200)
        pp = PlanProjector()
        pp.horizont = 300
        pp.set_theta(theta)
        points = pp.get_points()
        p = points[0]
        (lvp, dvp, rvp) = pp.get_vps()
        
        projpoints = self.project_plan(points, pp, lvp, rvp)
        unit_square = self.build_square(projpoints, lvp, rvp)
        big_square = self.double_square(unit_square, lvp, rvp)
        big_top = self.build_cube(big_square, 200, lvp, rvp)
        big_square = [x.plus(shift) for x in big_square]
        big_top = [x.plus(shift) for x in big_top]
        self.draw_wireframe_cube(cr, big_square, big_top)

    def corner_test(self, widget, data):
        now = time.time()
        duration = 500
        fraction = (int(100*now) % duration)/float(duration)
        theta = math.pi/2.0*fraction
        cr = self.canvas.window.cairo_create()
        cr.set_operator(cairo.OPERATOR_SOURCE)
        cr.set_source_rgb(0, 0, 0)
        shift = Point(0, -200)
        pp = PlanProjector()
        pp.horizont = 300
        pp.set_theta(theta)
        points = pp.get_points()
        p = points[0]
        (lvp, dvp, rvp) = pp.get_vps()
        
        projpoints = self.project_plan(points, pp, lvp, rvp)
        unit_square = self.build_square(projpoints, lvp, rvp)
        big_square = self.double_square(unit_square, lvp, rvp)
        big_top = self.build_cube(big_square, 200, lvp, rvp)
        points = [big_top[0], big_top[1], big_top[-1], big_square[0], big_square[1], big_square[-1]]
        points = [Point(p.x, 800 - p.y) for p in points]
        self.draw_corner_cube(cr, points, Point(lvp.x, 800-lvp.y), Point(rvp.x, 800-rvp.y), 800 - pp.horizont)

    def fill_test(self, widget, data):
        now = time.time()
        duration = 500
        fraction = (int(100*now) % duration)/float(duration)
        theta = math.pi/2.0*fraction
        cr = self.canvas.window.cairo_create()
        cr.set_operator(cairo.OPERATOR_SOURCE)
        cr.set_source_rgb(0, 0, 0)
        shift = Point(0, -200)
        pp = PlanProjector()
        pp.horizont = 300
        pp.set_theta(theta)
        points = pp.get_points()
        p = points[0]
        (lvp, dvp, rvp) = pp.get_vps()
        
        projpoints = self.project_plan(points, pp, lvp, rvp)
        unit_square = self.build_square(projpoints, lvp, rvp)
        big_square = self.double_square(unit_square, lvp, rvp)
        big_top = self.build_cube(big_square, 200, lvp, rvp)
        big_square = [x.plus(shift) for x in big_square]
        big_top = [x.plus(shift) for x in big_top]
        shade = 0.8
        self.draw_filled_cube(cr, big_square, big_top, shade*fraction, shade*(1.0-fraction), 0.7)


    def draw_square(self, cr, points):
        cr.move_to(points[0].x, points[0].y)
        for p in points[1:]:
            cr.line_to(p.x, p.y)
        cr.close_path()
        cr.stroke()
        
    def draw_dot_cube(self, cr, bottom, top):
        for p in bottom + top:
            cr.rectangle(p.x-2, p.y-2, 4, 4)
        cr.fill()
    
    def draw_wireframe_cube(self, cr, bottom, top):
        self.draw_square(cr, bottom)
        self.draw_square(cr, top)
        for i in range(len(bottom)):
            b = bottom[i]
            t = top[i]
            cr.move_to(b.x, b.y)
            cr.line_to(t.x, t.y)
        cr.stroke()
        
    def set_color(self, cr, c):
        try:
            cr.set_source_rgb(c[0], c[1], c[2])
        except TypeError:
            cr.set_source_rgb(c, c, c)

    def draw_filled_cube(self, cr, bottom, top, rightcolor, leftcolor, topcolor):
        if bottom[0].x < bottom[1].x:
            self.set_color(cr, rightcolor)
            cr.move_to(bottom[0].x, bottom[0].y)
            cr.line_to(top[0].x, top[0].y)
            cr.line_to(top[1].x, top[1].y)
            cr.line_to(bottom[1].x, bottom[1].y)
            cr.close_path()
            cr.fill()

        if bottom[0].x > bottom[3].x:
            self.set_color(cr, leftcolor)
            cr.move_to(bottom[0].x, bottom[0].y)
            cr.line_to(top[0].x, top[0].y)
            cr.line_to(top[3].x, top[3].y)
            cr.line_to(bottom[3].x, bottom[3].y)
            cr.close_path()
            cr.fill()
            
        if top[0].y > top[2].y:
            self.set_color(cr, topcolor)
            cr.move_to(top[0].x, top[0].y)
            cr.line_to(top[1].x, top[1].y)
            cr.line_to(top[2].x, top[2].y)
            cr.line_to(top[3].x, top[3].y)
            cr.close_path()
            cr.fill()
            
        if bottom[0].y < bottom[2].y:
            self.set_color(cr, topcolor)
            cr.move_to(bottom[0].x, bottom[0].y)
            cr.line_to(bottom[1].x, bottom[1].y)
            cr.line_to(bottom[2].x, bottom[2].y)
            cr.line_to(bottom[3].x, bottom[3].y)
            cr.close_path()
            cr.fill()
        
    def draw_corner_cube(self, cr, points, lvp, rvp, horizont, leftcolor=None, rightcolor=None, topcolor=None):
        assert(len(points) == 6)
        if rightcolor is not None:
            self.set_color(cr, rightcolor)
        if points[0].x < points[1].x:
            cr.move_to(points[0].x, points[0].y)
            cr.line_to(points[1].x, points[1].y)
            cr.line_to(points[4].x, points[4].y)
            cr.line_to(points[3].x, points[3].y)
            cr.close_path()
            if rightcolor is None:
                cr.stroke()
            else:
                cr.fill()

        if leftcolor is not None:
            self.set_color(cr, leftcolor)
        if points[0].x > points[2].x:
            cr.move_to(points[0].x, points[0].y)
            cr.line_to(points[2].x, points[2].y)
            cr.line_to(points[5].x, points[5].y)
            cr.line_to(points[3].x, points[3].y)
            cr.close_path()
            if leftcolor is None:
                cr.stroke()
            else:
                cr.fill()
        
        if points[3].y > horizont:
            if topcolor is not None:
                self.set_color(cr, topcolor)
            back = line_intersection(points[4], lvp, points[5], rvp)
            cr.move_to(points[4].x, points[4].y)
            cr.line_to(back.x, back.y)
            cr.line_to(points[5].x, points[5].y)
            cr.line_to(points[3].x, points[3].y)
            cr.close_path()
            if topcolor is None:
                cr.stroke()
            else:
                cr.fill()
        
        elif points[0].y < horizont:
            if topcolor is not None:
                self.set_color(cr, topcolor)
            back = line_intersection(points[1], lvp, points[2], rvp)
            cr.move_to(points[1].x, points[1].y)
            cr.line_to(back.x, back.y)
            cr.line_to(points[2].x, points[2].y)
            cr.line_to(points[0].x, points[0].y)
            cr.close_path()
            if topcolor is None:
                cr.stroke()
            else:
                cr.fill()
    def draw_corner_cube_shadow(self, cr, points, sun, sun_ground):
        shadow_points = []
        horizont = sun_ground.y
        shadow_points.append(points[1])
        shadow_points.append(line_intersection(sun, points[4], sun_ground, points[1]))
        shadow_points.append(line_intersection(sun, points[3], sun_ground, points[0]))
        shadow_points.append(line_intersection(sun, points[5], sun_ground, points[2]))
        shadow_points.append(points[2])
        shadow_points.append(points[0])
        cr.move_to(shadow_points[0].x, shadow_points[0].y)
        for p in shadow_points[1:]:
            cr.line_to(p.x, p.y)
        cr.close_path()
        cr.fill()

    def project_plan(self, points, pp, lvp, rvp):
        ground_corner = Point(pp.station.x, pp.ground)
        pplane1 = Point(0, pp.picture_plane)
        pplane2 = Point(1000, pp.picture_plane)
        projpoints = []
        for p in points:
            temppoint = line_intersection(pp.station, p, pplane1, pplane2)
            down = Point(temppoint.x, temppoint.y + 100)
            if temppoint.x < pp.station.x:
                vp = lvp
            else:
                vp = rvp
            projpoint = line_intersection(temppoint, down, ground_corner, vp)
            projpoints.append(projpoint)
        return projpoints
        
    def build_square(self, points, lvp, rvp):
        assert(len(points) == 3)
        p3 = line_intersection(points[1], lvp,  points[2], rvp)
        return [points[0], points[1], p3, points[2]]
        
    def build_cube(self, points, height, lvp, rvp):
        p0 = Point(points[0].x, points[0].y - height)
        up = []
        for p in points:
            up.append(Point(p.x, p.y - 10)) 
        p1 = line_intersection(points[1], up[1], p0, rvp)
        p3 = line_intersection(points[3], up[3], p0, lvp)
        p2 = line_intersection(p1, lvp, p3, rvp)
        return [p0, p1, p2, p3]
        
    def double_square(self, square, lvp, rvp):
        p2 = square[2]
        middle1 = extend_line(square[1], square[0])
        middle2 = extend_line(square[3], square[0])
        p3 = line_intersection(middle1, lvp, square[2], square[3])
        p1 = line_intersection(middle2, rvp, square[1], square[2])
        p0 = line_intersection(middle2, rvp, middle1, lvp)
        return [p0, p1, p2, p3]

    def key_pressed(self, widget, key):
        if key.keyval == gtk.keysyms.q or key.keyval == gtk.keysyms.Escape:
            gtk.main_quit()

def main():
    if len(sys.argv) > 1:
        fullscreen = True
    else:
        fullscreen = False
    app = TwoApp(fullscreen)
    gtk.main()


if __name__ == '__main__':
    main()
